;* * * * * * * * * * * * * * * * * * 
; CHESS.ASM - To illustrate 48k RamFont mode of the GB112 model HGC
;* * * * * * * * * * * * * * * * * *

;EQUATES

buffer_start	equ	0b000h		; Beginning of HGC video buffer

scrn_on         equ	28h		; Display mode port (3B8) - bits 5 + 3
scrn_off_text	equ	20h		; D_M_C port - disable video

index_reg	equ	03b4h		; 6845 pointer to data register
data_reg	equ	03b5h		; 6845 data register
dmc_port	equ	03b8h		; Display Mode Control Port
status_port	equ	03bah		; Display Status Port
xmode_reg	equ	14h		; GB112 RamFont Mode Control Register
score_reg	equ	15h		; For Placement of Underline attribute
strike_reg	equ	16h		; For Placement of Overstrike attribute

id_mask		equ	00110000b	; GB112 detection
id_code		equ	00010000b	; GB112 detection

blank_cnt	equ	3fffh		; Video buffer below RamFont storage
font_seg	equ	0b400h		; RamFont character storage area

open_file	equ	3dh		; DOS function numbers
print_string	equ	9
close_file	equ	3eh
read_file	equ	3fh

size_of_file	equ	28672		; Font file length - for LOAD routine

ramfnt		equ	1		; xModeReg control bits
fnt_48k         equ	4
eight_wide	equ	2
nine_wide	equ	0
fnt_4k		equ	0

scrn_on_b	equ	28h		; Set blink bit in D_M_C port
scrn_on_nb	equ	8		; Clear blink bit in D_M_C port

pause_period	equ	500		; For killing time

start_of_board	equ	368		; Upper left of chess board
cr_lf		equ	442		; # of bytes to next row of board

title_pos	equ	72		; Positions for text strings
move_pos	equ	1216
i_pos		equ	2652
i_1_pos		equ	2828
i_2_pos		equ	3012
i_3_pos		equ	3192
i_4_pos		equ	3372
i_5_pos		equ	3552
i_6_pos		equ	3728
i_7_pos		equ	3908
white_label_pos equ	4870
black_label_pos equ	190

; word attribute equates for 48k RamFont mode - used by PUT routine

normal		equ	0000h		; to OR with character
high_intensity	equ	8000h		; when blink enable bit is set
boldface	equ	8000h		; when blink enable bit is clear
blink		equ	4000h		; when blink enable bit is set
reverse         equ	4000h           ; when blink enable bit is clear
overstrike	equ	2000h
underline	equ	1000h

; byte attributes - used by WRITE_STRING routine

db_normal	equ	00h
db_high_intens	equ	80h
db_boldface	equ	80h
db_blink	equ	40h
db_reverse	equ	40h
db_overstrike	equ	20h
db_underline	equ	10h

white		equ	0			; square color
black		equ	1

white_color_move	equ	00000000b	; player
black_color_move	equ	11111111b

; Character code of chess pieces in RamFont. This is the offset into RamFont
; storage where the individual chess pieces are held.

black_space		equ	101h
white_space		equ	116h
black_pawn_black	equ	12bh
black_pawn_white	equ	140h
white_pawn_black	equ	155h
white_pawn_white	equ	16ah
black_castle_black	equ	17fh
black_castle_white	equ	194h
white_castle_black	equ	1a9h
white_castle_white	equ	1beh
black_knight_black	equ	1d3h
black_knight_white	equ	1e8h
white_knight_black	equ	1fdh
white_knight_white	equ	212h
black_bishop_black	equ	227h
black_bishop_white	equ	23ch
white_bishop_black	equ	251h
white_bishop_white	equ	266h
black_queen_black	equ	27bh
black_queen_white	equ	290h
white_queen_black	equ	2a5h
white_queen_white	equ	2bah
black_king_black	equ	2cfh
black_king_white	equ	2e4h
white_king_black	equ	2f9h
white_king_white	equ	30eh
bottom_of_board		equ	323h


; Screen addresses - offsets into the video buffer where chess pieces will
; appear on the screen.

a1	equ	368
b1	equ	382
c1	equ	396
d1	equ	410
e1	equ	424
f1	equ	438
g1	equ	452
h1	equ	466
a2	equ	908
b2	equ	922
c2	equ	936
d2	equ	950
e2	equ	964
f2	equ    	978
g2	equ    	992
h2	equ    	1006
a3	equ    	1448
b3	equ    	1462
c3	equ    	1476
d3	equ    	1490
e3	equ    	1504
f3	equ    	1518
g3	equ    	1532
h3	equ    	1546
a4	equ    	1988
b4	equ    	2002
c4	equ    	2016
d4	equ    	2030
e4	equ    	2044
f4	equ    	2058
g4	equ    	2072
h4	equ    	2086
a5	equ    	2528
b5	equ    	2542
c5	equ    	2556
d5	equ    	2570
e5	equ    	2584
f5	equ    	2298
g5	equ    	2612
h5	equ    	2626
a6	equ    	3068
b6	equ    	3082
c6	equ    	3096
d6	equ    	3110
e6	equ    	3124
f6	equ    	3138
g6	equ    	3152
h6	equ    	3166
a7	equ    	3608
b7	equ    	3622
c7	equ    	3636
d7	equ    	3650
e7	equ    	3664
f7	equ    	3678
g7	equ    	3692
h7	equ    	3706
a8	equ    	4148
b8	equ    	4162
c8	equ    	4176
d8	equ    	4190
e8	equ    	4204
f8	equ    	4218
g8	equ    	4232
h8	equ    	4246
bottom_line	equ    	4688


check_key	macro
	mov	ah,1
	int	16h
	endm

get_key	macro
	mov	ah,0
	int	16h
	endm


cseg segment
	assume cs:cseg,ds:cseg,es:nothing
	org	100h
start:
	jmp	real_start

title_string		db	18,'RamFont Chessboard'
white_label		db	5,'WHITE'
black_label		db	5,'BLACK'
white_turn		db	13,'             '
black_turn		db	13,'             '
mate			db	13,'  CHECKMATE  '
instructions_5		db	16,'R - Resets board'
instructions_6		db	22,'S - Scrolls the screen'
instructions_7		db	19,'ESC - Exits program'
instructions_8		db	16,'P - Plays a game'
instructions_9		db      15,'   Instructions'

attribute		dw	?			; word attribute
char_att		db	db_normal		; byte attribute

error_flag		db	0
file_error		db	0dh,0ah,'Unable to open font file.',0dh,0ah,24h
current_square		dw	start_of_board

old_square		dw	?
selected_square		dw	?

exit_flag		db	0
board_error_flag	db	0

first_enter		db	0
enter_flag		db	0

play_loop		db	0
move_number		dw	0
play_pos		dw	0

color			db	white
old_color		db	?
move_color		db	white

selected_piece		dw	?
piece_to_move		dw	?
current_ptr		dw	0

board_table		dw	64 dup (?)

piece_table		dw	white_pawn_white
			dw	white_pawn_black	
			dw	black_pawn_white	
			dw	black_pawn_black	
			dw	white_castle_white	
			dw	white_castle_black	
			dw	black_castle_white	
			dw	black_castle_black	
			dw	white_knight_white	
			dw	white_knight_black	
			dw	black_knight_white	
			dw	black_knight_black	
			dw	white_bishop_white	
			dw	white_bishop_black	
			dw	black_bishop_white	
			dw	black_bishop_black	
			dw	white_queen_white	
			dw	white_queen_black	
			dw	black_queen_white	
			dw	black_queen_black	
			dw	white_king_white	
			dw	white_king_black	
			dw	black_king_white	
			dw	black_king_black	


real_start:
	push	cs
	pop	ds

	call	test_board		; See if we have a HGC model GB112
	cmp	board_error_flag,0
	je	RamFont_ok

	ret				; ABORT - don't have GB112

; The code that follows illustrates what needs to be done to set the GB112
; into RamFont mode : Program the 6845 for the desired character size, set
; the xModeReg with proper parameters, and set the overstrike and underscore
; registers (even though we won't use these attributes with the chess
; program).


RamFont_ok:
	mov	al,1
	mov	bx,0000			; BLANKING VALUE - this works because
	call	prog_6845		; CHESS.FNT character at code 000 is
					; blank. The value 0 specifies normal
					; attribute for 48K RamFont mode.

	mov	dx,index_reg
	mov	al,xmode_reg
	out	dx,al

; select 48k, 8 dot
	mov	dx,data_reg
	mov	al,00000111b		;RamFont-ON, 48k-ON, 8 dot-ON
	out	dx,al

; set underline and overstrike positions

	mov	dx,index_reg
	mov	al,score_reg
	out	dx,al

	mov	dx,data_reg
	mov	al,11
	out	dx,al			; underscore at height - 1

	mov	dx,index_reg
	mov	al,strike_reg
	out	dx,al

	mov	dx,data_reg
	mov	al,6
	out	dx,al			; overstrike at height / 2

; Load font file into RamFont storage

	call	load
	jnc	go_on

	mov	error_flag,1
	jmp	normal_return

go_on:
	mov	dx,dmc_port		;turn the screen on
	mov	al,scrn_on_b
	out	dx,al

; This ends the GB112 RamFont initialization portion of the code.

; Set up the chess board screen

	mov	bx,buffer_start
	mov	es,bx

	mov	attribute,normal
	call	reset_board

	mov	si,offset title_string
	mov	di,title_pos
	call	write_string

	mov	si,offset white_turn
	mov	di,move_pos
	call	write_string
	
	mov	si,offset instructions_9
	mov	di,i_pos
	call	write_string

	mov	si,offset instructions_8
	mov	di,i_2_pos
	call	write_string

	mov	si,offset instructions_5
	mov	di,i_3_pos
	call	write_string

	mov	si,offset instructions_6
	mov	di,i_4_pos
	call	write_string

	mov	si,offset instructions_7
	mov	di,i_5_pos
	call	write_string

	mov	si,offset white_label
	mov	di,white_label_pos
	call	write_string

	mov	si,offset black_label
	mov	di,black_label_pos
	call	write_string


;*.*.*.*. MAIN LOOP *.*.*.*.

get_input:
	mov	al,play_loop
	cmp	al,1
	je	play

	check_key
	jz	get_input

	get_key
	cmp	ah,19h				; P key - do we want to 
	je	g_i_1				; play the automatic game?

	jmp	g_i_2				; No - we will look for 
						; other keys.
g_i_1:
	mov	al,1
	mov	play_loop,al
	mov	bx,offset cs:play_table
	mov	play_pos,bx

play:
	mov	cx,move_number
	cmp	cx,520
	je	end_play
	inc	cx
	mov	move_number,cx
	mov	bx,play_pos
	mov	ah,cs:[bx]
	inc	bx
	mov	play_pos,bx

	call	pause

	call	process_input

	jmp	g_i_3
			
end_play:
	mov	al,0
	mov	play_loop,al
	mov	cx,0
	mov	move_number,cx

	mov	si,offset mate
	mov	di,move_pos
	call	write_string
	
	jmp	get_input

; The components are here to implement a dumb chess board. However, for the
; sake of demonstrating the speed of RamFont mode, this table of keystrokes
; exists to automate a game. If you want to use the existing routines to  
; check for keystrokes and move accordingly, please do.

play_table	db	4dh,50h,50h,50h,50h,50h,50h,4dh,4dh,1ch,48h,48h,1ch
		db	48h,48h,48h,1ch,50h,50h,1ch,4dh,50h,50h,50h,1ch,48h
		db	48h,1ch,48h,4bh,1ch,4dh,50h,1ch,50h,50h,50h,4bh,4bh
		db	4bh,1ch,48h,48h,4dh,1ch,48h,48h,48h,48h,48h,4bh,1ch
		db	50h,50h,4dh,1ch,4dh,4dh,4dh,50h,50h,50h,50h,50h,1ch
		db	48h,4bh,1ch,48h,48h,48h,48h,48h,1ch,50h,1ch,50h,50h
		db	50h,4bh,4bh,1ch,48h,4dh,4dh,1ch,4dh,4dh,48h,48h,48h
		db	48h,1ch,50h,50h,4bh,1ch,50h,50h,50h,4bh,4bh,4bh,50h
		db	50h,1ch,48h,48h,4dh,4dh,4dh,4dh,48h,48h,1ch,48h,48h
		db	4dh,1ch,50h,1ch,50h,4bh,1ch,48h,4bh,1ch,48h,4dh,1ch
		db	50h,4bh,1ch,50h,50h,50h,50h,4bh,1ch,48h,4dh,48h,4dh
		db	48h,4dh,1ch,4bh,48h,48h,48h,4bh,1ch,4dh,50h,1ch,50h
		db	50h,50h,50h,50h,50h,1ch,48h,48h,4bh,1ch,48h,48h,48h
		db	1ch,50h,1ch,4bh,50h,1ch,4bh,4bh,48h,1ch,4bh,48h,48h
		db	1ch,50h,1ch,4dh,50h,1ch,50h,50h,4dh,1ch,48h,48h,48h
		db	48h,48h,4bh,1ch,50h,50h,4bh,4bh,1ch,4dh,4dh,4dh,4dh
		db	50h,50h,50h,50h,50h,1ch,4dh,4dh,1ch,'w',4dh,1ch,4bh
		db	4bh,1ch,4bh,48h,48h,48h,48h,48h,4bh,4bh,4bh,4bh,1ch
		db	4dh,4dh,4dh,50h,50h,50h,1ch,50h,50h,1ch,48h,48h,1ch
		db	48h,48h,48h,4bh,1ch,4dh,50h,50h,1ch,50h,1ch,48h,48h
		db	4bh,4bh,1ch,48h,48h,48h,4dh,4dh,4dh,1ch,4dh,1ch,50h
		db	50h,50h,50h,50h,50h,50h,4bh,4bh,4bh,4bh,4bh,1ch,4dh
		db	4dh,4dh,1ch,48h,48h,48h,1ch,4bh,4bh,48h,1ch,4dh,4dh
		db	50h,50h,50h,50h,1ch,48h,48h,48h,48h,48h,48h,48h,1ch
		db	4bh,4bh,4bh,1ch,4dh,4dh,4dh,1ch,50h,50h,50h,50h,50h
		db	50h,4dh,4dh,4dh,1ch,48h,1ch,48h,48h,48h,48h,48h,4bh
		db	4bh,4bh,4bh,4dh,1ch,50h,50h,50h,50h,50h,48h,48h,1ch
		db	4dh,4dh,4dh,50h,50h,1ch,48h,1ch,48h,4bh,1ch,4dh,50h
		db	1ch,48h,4dh,1ch,50h,4bh,1ch,48h,48h,48h,1ch,50h,50h
		db	50h,48h,48h,4bh,1ch,50h,50h,50h,50h,4bh,4bh,4bh,1ch
		db	48h,1ch,4dh,4dh,4dh,4dh,48h,48h,48h,48h,48h,4dh,1ch
		db	4bh,1ch,50h,50h,50h,50h,50h,4bh,4bh,4bh,4bh,1ch,48h
		db	1ch,4dh,4dh,4dh,4dh,48h,48h,48h,48h,1ch,50h,50h,50h
		db	50h,1ch,50h,50h,50h,1ch,4dh,1ch,48h,48h,48h,48h,4bh
		db	4bh,4bh,4bh,1ch,50h,50h,1ch,4dh,4dh,1ch,48h,48h,4dh
		db	1ch,48h,4bh,1ch,4dh,50h,1ch,50h,4bh,4bh,4bh,4bh,1ch
		db	48h,4bh,1ch,4dh,4dh,4dh,4dh,4dh,1ch,50h,4bh,1ch,50h
		db	50h,1ch,48h,1ch,4dh,48h,1ch,4dh,1ch,50h,50h,50h,4bh
		db	4bh,1ch,48h,1ch,48h,4bh,4bh,1ch,50h,50h,1ch,4dh,4dh
		db	4dh,4dh,1ch,48h,4bh,1ch,4dh,48h,48h,1ch,50h,50h,1ch


g_i_2:
	call	process_input			; which acts on keystrokes
g_i_3:
	mov	al,exit_flag			; on return is exit flag set?
	cmp	al,0
	je	keep_going

	jmp	normal_return			; if so - back to DOS
	
keep_going:
	mov	al,enter_flag			;are we handling a strike
	cmp	al,1				;of the enter key?
	jne	k_g_1

	mov	al,first_enter			;yes - is it the first strike?
	cmp	al,1
	je	enter_first

	jmp	enter_second

k_g_1:
	mov	dx,old_square			;must be a cursor key move
	mov	bx,normal
	call	put_att

	mov	dx,current_square
	mov	bx,high_intensity
	call	put_att

enter_first:					;go back for more
	mov	enter_flag,0
	jmp	get_input

enter_second:					;second enter strike
	mov	ax,selected_piece		;move pieces
	mov	dx,current_square
	mov	attribute,high_intensity
	call	put

	mov	al,old_color
	cmp	al,white
	jne	b_s
	mov	ax,white_space
	jmp	place_square
b_s:
	mov	ax,black_space

place_square:
	mov	dx,selected_square
	mov	attribute,normal
	call	put

	mov	al,move_color
	not	al
	mov	move_color,al

	cmp	al,white_color_move
	jne	its_black
	mov	si,offset white_turn
	mov	di,move_pos
	call	write_string

	jmp	way_back

its_black:
	mov	si,offset black_turn
	mov	di,move_pos
	call	write_string
way_back:
	mov	enter_flag,0

	jmp	get_input			;go back for more


;*.*.*.*.  END OF MAIN LOOP  *.*.*.*

; This following code illustrates what is necessary to set the GB112 back
; to text mode before returning to DOS.

normal_return:

; turn off RamFont mode

	mov	al,0
	mov	bx,0720h		; BLANKING VALUE - in the case of text
					; mode, normal attribute + ASCII space
	call	prog_6845

	mov	dx,index_reg		; set xModeReg
	mov	al,xmode_reg
	out	dx,al

	mov	dx,data_reg
	mov	al,0			; ROM generator, normal text mode
	out	dx,al

	mov	dx,dmc_port		; turn screen on, blink activated
	mov	al,scrn_on_b
	out	dx,al

; check error status

	mov	al,error_flag		; are we here due to load error?
	cmp	al,1
	jne	finish			; no, we're ok

	mov	ah,print_string		; yes, print message
	mov	dx,offset file_error
	int	21h

finish:
	ret


;* * * * * * * * * * * * * * * * * * 
; PROCESS_INPUT - control passed here to handle keystrokes. In spite of the
;                 on-screen instructions. This routine checks for CURSOR keys
;                 to move the high-intensity attribute marker, ENTER keys to 
;		  select and move a piece, and B and W keys to change players.
;* * * * * * * * * * * * * * * * * *
process_input	proc	near

	cmp	ah,48h
	jae	cursor_key

	cmp	ah,1ch			;enter key
	jne	test_black
	jmp	select_piece

test_black:
	cmp	ah,30h			;b key
	jne	test_white
	mov	al,black_color_move
	mov	move_color,al

	mov	si,offset black_turn
	mov	di,move_pos
	call	write_string

	ret

test_white:
	cmp	ah,11h			;w key
	jne	test_reset
	mov	al,white_color_move
	mov	move_color,al

	mov	si,offset white_turn
	mov	di,move_pos
	call	write_string

	ret

test_reset:
	cmp	ah,13h			;r key
	jne	test_scroll
	call	reset_board

	mov	al,white
	mov	color,al
	mov	move_color,al
	mov	ax,start_of_board
	mov	current_square,ax

	mov	current_ptr,0

	mov	attribute,high_intensity
	mov	si,offset white_turn
	mov	di,move_pos
	call	write_string

	ret

test_scroll:				;s key
	cmp	ah,1fh
	jne	test_escape
	jmp	to_scroll

test_escape:
	cmp	ah,1			;ESC key
	jne	no_good_keys
	jmp	escape

no_good_keys:				; no acceptable input, return
	ret				; doing nothing

cursor_key:
	mov	bp,current_ptr		; get offset into board_table
	mov	bx,current_square	; get offset into video buffer
	mov	cl,color		; get current square color

	cmp	ah,4dh			; cursor right
	jne	cursor_left
	cmp	bx,h1			; see if we are at the extremes of
	je	edge			; the board
	cmp	bx,h2			; if so, return doing nothing
	je	edge
	cmp	bx,h3
	je	edge
	cmp	bx,h4
	je	edge
	cmp	bx,h5
	je	edge
	cmp	bx,h6
	je	edge
	cmp	bx,h7
	je	edge
	cmp	bx,h8
	je	edge
					; if we can move in this direction:
	mov	old_square,bx		; store our location
	add	bx,14			; add bytes for proper direction move
	mov	current_square,bx
	add	bp,2			; add bytes for board_table equivalent
	mov	current_ptr,bp
	not	cl			; we moved, compliment color variable
	mov	color,cl		; (color alternates with every move )
edge:
	ret				; go back


cursor_left:				; And so on, as above, for the other
	cmp	ah,4bh			; three possible directions.
	jne	cursor_up
	cmp	bx,a1
	je	edge1
	cmp	bx,a2
	je	edge1
	cmp	bx,a3
	je	edge1
	cmp	bx,a4
	je	edge1
	cmp	bx,a5
	je	edge1
	cmp	bx,a6
	je	edge1
	cmp	bx,a7
	je	edge1
	cmp	bx,a8
	je	edge1

	mov	old_square,bx
	sub	bx,14
	mov	current_square,bx
	sub	bp,2
	mov	current_ptr,bp
	not	cl
	mov	color,cl

edge1:
	ret

cursor_up:
	cmp	ah,48h
	jne	cursor_down
	cmp	bx,a1
	je	edge2
	cmp	bx,b1
	je	edge2
	cmp	bx,c1
	je	edge2
	cmp	bx,d1
	je	edge2
	cmp	bx,e1
	je	edge2
	cmp	bx,f1
	je	edge2
	cmp	bx,g1
	je	edge2
	cmp	bx,h1
	je	edge2

	mov	old_square,bx
	sub	bx,540
	mov	current_square,bx
	sub	bp,16
	mov	current_ptr,bp
	not	cl
	mov	color,cl
edge2:
	ret


cursor_down:
	cmp	ah,50h
	jne	edge3
	cmp	bx,a8
	je	edge3
	cmp	bx,b8
	je	edge3
	cmp	bx,c8
	je	edge3
	cmp	bx,d8
	je	edge3
	cmp	bx,e8
	je	edge3
	cmp	bx,f8
	je	edge3
	cmp	bx,g8
	je	edge3
	cmp	bx,h8
	je	edge3

	mov	old_square,bx
	add	bx,540
	mov	current_square,bx
	add	bp,16
	mov	current_ptr,bp
	not	cl
	mov	color,cl
edge3:
	ret

select_piece:
	cmp	first_enter,1			; if first_enter flag is set
	je	second_enter

	mov	bp,current_ptr			
	mov	ax,word ptr board_table[bp]	; see if a piece occupies 
	cmp	ax,50				; this square
	jb	continue			; if no - illegal selection
	
	ret

continue:
	mov	piece_to_move,ax		; initialize piece type
	mov	ah,color			; get color of current square
	mov	old_color,ah			; save it 
	mov	word ptr board_table[bp],50	; update board_table

	mov	ax,current_square		
	mov	selected_square,ax		; initialize variable

	mov	enter_flag,1			; so we know we've been here
	mov	first_enter,1

	ret

second_enter:
	mov	ax,piece_to_move		; get piece type
	mov	word ptr board_table[bp],ax	; update board_table
	mov	bp,piece_to_move		
	mov	ah,move_color			; add for proper offset
	cmp	ah,white_color_move		; into piece_table
	je	around
	add	bp,4
around:
	mov	ah,color
	cmp	ah,white
	je	around_1
	add	bp,2
around_1:
	mov	bx,word ptr piece_table[bp]
	mov	selected_piece,bx		; This will be the piece to 
	mov	enter_flag,1			; place indicate that we are
	mov	first_enter,0			; returning from second
						; enter strike.
	ret

to_scroll:
	call	scroll
	ret

escape:
	mov	exit_flag,0ffh			; indicate that keystroke was
	ret					; to quit

process_input	endp
		

;* * * * * * * * * * * * * * * * * * * * *
; PUT_ATT - Places desired attribute at specified square
;
; On Entry - DX = address of screen position
;	     BX = desired attribute
;* * * * * * * * * * * * * * * * * * * * * 

put_att	proc	near
	xor	cx,cx
	mov	di,dx			;DI = screen address

	mov	cx,3			;chess component is 3 characters high
p_a_1:
	push	cx	
	mov	cx,7			;chess component is 7 bytes wide
p_a_2:
	mov	ax,es:[di]
	and	ax,0000111111111111b	;clear attribute bits
	or	ax,bx
	mov	es:[di],ax
	add	di,2
	loop	p_a_2

	add	di,166			;"carriage return, line feed"
	pop	cx
	loop	p_a_1

	ret

put_att	endp

;* * * * * * * * * * * * * * * * * * * * * 
; PUT - Will put the RamFont chess piece into the text buffer with specified
;	attribute.
; On Entry - AX = character code 
;            DX	= address of screen position
;	     ATTRIBUTE variable initialized
;* * * * * * * * * * * * * * * * * * * * *

put	proc	near
	push	cx
	xor	cx,cx
	xor	bx,bx
	mov	bx,attribute

	mov	di,dx			; DI = address in screen buffer

	mov	cx,3
p_1:
	push	cx
	mov	cx,7			; chess piece is 7 bytes wide
p_2:
	or	ax,bx			; highest 4 bits comprise attribute
	mov	es:[di],ax		; attribute/character code to buffer

	add	di,2			; point to next screen location
	inc	ax			; point to next character
	loop	p_2

	add	di,166			; (180-14) gets to buffer address of
					; next portion of character
	pop	cx
	loop	p_1

	pop	cx
	ret

put	endp

;* * * * * * * * * * * * * * * * * * *
; WRITE_STRING - Maps a character and attribute to RamFont buffer
;	On Entry - SI = pointer to Count:string to write
;		   DI = address offset into screen buffer
;		   attribute variable initialized
;* * * * * * * * * * * * * * * * * * *
write_string	proc	near
	push	cx
	xor	cx,cx

	mov	al,char_att
	mov	cl,byte ptr cs:[si]
	inc	si
p_c_1:
	mov	bl,byte ptr cs:[si]
	or	bh,al
	mov	es:[di],bx
	add	di,2
	inc	si
	loop	p_c_1

	pop	cx
	ret

write_string	endp

;* * * * * * * * * * * * * * * * * * *
; PAUSE - Kills some time, so the game playing doesn't go by so fast.
;* * * * * * * * * * * * * * * * * * * 

pause	proc	near

	push	cx
	push	bx

	xor	bx,bx
	mov	cx,pause_period

kill_time:
	inc	bx
	loop	kill_time

	pop	bx
	pop	cx

	ret

pause	endp	
	
;* * * * * * * * * * * * * * * * * * * 
; RESET_BOARD - Redraws the chess board, does NOT clear screen.
;* * * * * * * * * * * * * * * * * * * 

reset_table		dw	194h,1d3h,23ch,27bh,2e4h,227h,1e8h,17fh
			dw	12bh,140h,12bh,140h,12bh,140h,12bh,140h
			dw	116h,101h,116h,101h,116h,101h,116h,101h
			dw	101h,116h,101h,116h,101h,116h,101h,116h
			dw	116h,101h,116h,101h,116h,101h,116h,101h
			dw	101h,116h,101h,116h,101h,116h,101h,116h
			dw	16ah,155h,16ah,155h,16ah,155h,16ah,155h
			dw	1a9h,212h,251h,2bah,2f9h,266h,1fdh,1beh
			dw	323h,323h,323h,323h,323h,323h,323h,323h

reset_board_table 	dw	8,16,24,32,40,24,16,8
			dw	0,0,0,0,0,0,0,0
			dw	50,50,50,50,50,50,50,50
			dw	50,50,50,50,50,50,50,50
			dw	50,50,50,50,50,50,50,50
			dw	50,50,50,50,50,50,50,50
			dw	0,0,0,0,0,0,0,0
			dw	8,16,24,32,40,24,16,8

reset_board	proc	near

	push	bp
	mov	dx,start_of_board
	mov	bp,0
	mov	ax,normal
	mov	attribute,ax

	mov	cx,9
r_b_1:
	push	cx
	mov	cx,7
r_b_2:
	mov	ax,reset_table[bp]
	call	put

	add	dx,14
	add	bp,2
	loop	r_b_2

	mov	ax,reset_table[bp]
	call	put

	add	dx,cr_lf
	add	bp,2
	pop	cx
	loop	r_b_1

	push	es
	push	ds
	pop	es

	mov	si,offset reset_board_table
	mov	di,offset board_table
	mov	cx,64

rep	movsw

	pop	es
	pop	bp

   	mov	si,offset white_label
	mov	di,white_label_pos
	call	write_string


	ret

reset_board	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * 
; PROG_6845 - This procedure programs the 6845 with one of two data tables.
; On Entry = AL = 0 for normal text mode.
;	     AL = 1 for 8 wide, 12 high RamFont mode.
;	     BX = blanking value, used to clear buffer
;* * * * * * * * * * * * * * * * * * * * * * * * * *

text_table	db	61h,50h,52h,0fh,19h,6,19h,19h,2,0dh,0bh,0ch,0,0
ram_font_table	db	107,90,92,15,29,10,29,29,2,0bh,0dh,0eh,0,0

prog_6845	proc	near

	push	bx		; save blanking value

	cmp	al,0		; select 6845 table
	jne	set_8_bit

	mov	si,offset text_table
	jmp	screen_off

set_8_bit:
      	mov	si,offset ram_font_table

screen_off:
	mov	al,scrn_off_text
	mov	dx,dmc_port
	out	dx,al

	mov	dx,03bfh		; Set the HGC+ to HALF, CHESS.FNT does
	mov	al,1			; not need to load into types above T4.
	out	dx,al			; If it did, the FULL configuration
					; would be necessary.
;program the 6845	
	mov	dx,index_reg
	mov	cx,14

	xor	ah,ah			; start from register 0

load_table:
	mov	al,ah
	out	dx,al

	inc	dx			; point to data register
	lodsb
	out	dx,al

	inc	ah			; point to next register
	dec	dx			; point again to index register
	
	loop	load_table

; clear the text buffer, but leave the screen off

	mov	ax,buffer_start
	mov	es,ax
	xor	di,di
	mov	cx,blank_cnt
	pop	ax			; get blanking value

rep	stosw

	ret

prog_6845	endp

;* * * * * * * * * * * * * * * * * 
; SCROLL - scrolls the chess board, to dramatize text mode
;* * * * * * * * * * * * * * * * * 
scroll	proc	near
	push	cx
	push	ds
	push	es
	pop	ds			; DS and ES point at screen buffer

	cld

	mov	cx,29
scroll_1:
	push	cx
	mov	di,5220			;# of columns * # of rows * 2
	mov	si,0000
	mov	cx,90

rep	movsw
	mov	di,0000
	mov	si,180
	mov	cx,2610			;# of columns * # of rows

rep	movsw
	pop	cx
	loop	scroll_1

	pop	ds
	pop	cx

	ret

scroll	endp

;* * * * * * * * * * * * * * * * *
; TEST_BOARD - Performs two tests - first, makes sure that a Hercules
;	       Graphics Card is present, then checks that it is a GB112.
;
; RETURNS -  board_error_flag = 1 if GB112 is not detected
;* * * * * * * * * * * * * * * * *

fail_message1	db	0dh,0ah,'ERROR - Hercules Graphics Card not present '
		db	'or not presently active.',0dh,0ah,'$'
fail_message2	db	0dh,0ah,'ERROR - Model GB112 Hercules Graphics Card '
		db	'not detected in system.',0dh,0ah,'$'

test_board	proc	near

	mov	ah,15			; check current video state
	int	10h			
	
	cmp	al,7			; is it 80 x 25 "B&W" Card ?
	je	mono_ok	   		; yes - continue

	mov	dx,offset fail_message1 ; no - quit
	jmp	test_failed

mono_ok:		       	
	mov	dx,status_port		; record state 
	in	al,dx			
	and	al,80h			; save bit 7 for test
	mov	ah,al

	mov	cx,8000h

examine:
	in	al,dx			; take another reading
	and	al,80h			; again save bit seven
	cmp	al,ah
	jne	hgc  			; if bit 7 changes, then it
	loop	examine			; is a Hercules Graphics Card
    
	mov	dx,offset fail_message1
	jmp	test_failed		; after this long it must be
					; something else - quit.
        
hgc:	in	al,dx			; test for GB112
	and	al,id_mask		; clear all but bits 4 and 5

	cmp	al,id_code		; test ID bits
	jne	t_f_1			; exit if failed - not a GB112

	ret				; normal return

t_f_1:
	mov	dx,offset fail_message2

test_failed:
	mov	ah,9
	int	21h

	mov	board_error_flag,1

	ret				; error return

test_board	endp			



;* * * * * * * * * * * * * * * * * 
; LOAD - Loads named character data file into RamFont
;
; RETURNS - Carry flag set if file error
;* * * * * * * * * * * * * * * * *
handle		dw	?
file_name	db	'chess.fnt',0

load	proc	near

	mov	dx,offset file_name
	mov	al,0
	mov	ah,open_file
	int	21h

	jnc	open_ok

	ret				;cannot load, return

open_ok:
	mov	handle,ax

	mov	ax,font_seg
	mov	es,ax
	xor	bp,bp

rd_char:
	mov	bx,handle
	mov	dx,bp
	mov	cx,size_of_file
	mov	ah,read_file
	push	es
	pop	ds
	int	21h

	push	cs
	pop	ds

done:
	mov	bx,handle
	mov	ah,close_file
	int	21h

	ret

load	endp

cseg	ends
	end	start

